﻿// ---------------------------------------------------------------
// <copyright file="ChromeHelper.cs" company="B33Rbaron">
//     Copyright (c) Daniel Birler.
//     Licensed under Microsoft Public License (Ms-PL).
// </copyright>
// <author>Daniel Birler</author>
// ---------------------------------------------------------------

namespace ChromeHost
{
  using System;
  using System.Diagnostics;
  using System.IO;
  using System.Windows.Forms;
  using Microsoft.Win32;

  /// <summary>
  /// A helper class for Google Chrome.
  /// </summary>
  public sealed class ChromeHelper
  {
    /// <summary>
    /// The registry key where the full executable path of chrome can be stored.
    /// </summary>
    private const string AppPathRegKey = @"Software\Microsoft\Windows\CurrentVersion\App Paths\chrome.exe";

    /// <summary>
    /// The path to Google Chrome from local application data.
    /// </summary>
    private const string Chrome = @"\Google\Chrome\Application\chrome.exe";

    /// <summary>
    /// The path to Google Chrome Custom.css from local application data.
    /// </summary>
    private const string CustomCss = @"\Google\Chrome\User Data\Default\User StyleSheets\Custom.css";

    /// <summary>
    /// The path for a temp Cusotm.css from local application data.
    /// </summary>
    private const string CustomCssTmp = @"\Google\Chrome\User Data\Default\User StyleSheets\_TMP_Custom.css";

    /// <summary>
    /// This field is responsible for initializing the ChromeHelper class.
    /// </summary>
    private static readonly Lazy<ChromeHelper> Lazy = new Lazy<ChromeHelper>(() => new ChromeHelper());

    /// <summary>
    /// Determines whether tiny scrollbars are installed or not.
    /// </summary>
    private bool isTinyScrollbarInstalled = false;

    /// <summary>
    /// Prevents a default instance of the ChromeHelper class from being created.
    /// </summary>
    private ChromeHelper()
    {
    }

    /// <summary>
    /// Gets the one and only instance of the ChromeHelper class.
    /// </summary>
    public static ChromeHelper Instance
    {
      get { return Lazy.Value; }
    }

    /// <summary>
    /// Returns if a chrome installation can be found.
    /// </summary>
    /// <returns>True if a chrome installation is found, otherwise false.</returns>
    public bool IsChromeInstalled()
    {
      return string.IsNullOrEmpty(this.FindChromePath()) ? false : true;
    }

    /// <summary>
    /// Runs a Google Chrome window in app mode.
    /// </summary>
    /// <param name="url">The url to start.</param>
    public void RunAsApp(Uri url)
    {
      string args = "--incognito -app=" + url.OriginalString;
      string chromePath = this.FindChromePath();

      if (!string.IsNullOrEmpty(chromePath))
      {
        ChromeWindowWatcher.Instance.Enabled = true;
        using (Process process = new Process())
        {
          process.StartInfo = new ProcessStartInfo(chromePath, args);
          process.Start();
        }
      }
    }

    /// <summary>
    /// Modifies Chrome Custom.css style sheet to use tiny scrollbars. If the file
    /// is not in the assumed location, chrome is not installed and this is skipped.
    /// </summary>
    [Obsolete("Chrome does not support Custom.css anymore.", true)]
    public void InstallTinyScrollBar()
    {
      if (this.isTinyScrollbarInstalled)
      {
        return;
      }

      string localApplicationDataPath = Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData);
      string customCssPath = localApplicationDataPath + ChromeHelper.CustomCss;
      string customCssPathTmp = localApplicationDataPath + ChromeHelper.CustomCssTmp;

      if (File.Exists(customCssPath))
      {
        if (File.Exists(customCssPathTmp))
        {
          DialogResult result = MessageBox.Show("Copy of Custom.css exists, delete and copy again?", "ChromeHost", MessageBoxButtons.YesNo, MessageBoxIcon.Question, MessageBoxDefaultButton.Button2, 0);
          if (result == DialogResult.Yes)
          {
            File.Delete(customCssPathTmp);
            File.Copy(customCssPath, customCssPathTmp);
          }
        }
        else
        {
          File.Copy(customCssPath, customCssPathTmp);
        }

        File.WriteAllText(customCssPath, Properties.Resources.Custom);
        this.isTinyScrollbarInstalled = true;
      }
    }

    /// <summary>
    /// Reverts Chrome Custom.css user style sheet.
    /// </summary>
    [Obsolete("Chrome does not support Custom.css anymore.", true)]
    public void UninstallTinyScrollBar()
    {
      if (!this.isTinyScrollbarInstalled)
      {
        return;
      }

      string localApplicationDataPath = Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData);
      string customCssPath = localApplicationDataPath + ChromeHelper.CustomCss;
      string customCssPathTmp = localApplicationDataPath + ChromeHelper.CustomCssTmp;

      string contents = string.Empty;
      if (File.Exists(customCssPathTmp))
      {
        contents = File.ReadAllText(customCssPathTmp);
        File.Delete(customCssPathTmp);
      }

      File.WriteAllText(customCssPath, contents);
      this.isTinyScrollbarInstalled = false;
    }

    /// <summary>
    /// Tries to find the path to the chrome executable.
    /// </summary>
    /// <returns>The found chrome path from the registry or a hard coded path to
    /// the current users local application data folder, or an empty string if
    /// none of the two return a file that exists.</returns>
    private string FindChromePath()
    {
      string chromePath = string.Empty;

      // Check from registry from CurrentUser
      using (RegistryKey chromeAppPathKey = Registry.CurrentUser.OpenSubKey(ChromeHelper.AppPathRegKey, RegistryKeyPermissionCheck.ReadSubTree))
      {
        if (chromeAppPathKey != null)
        {
          chromePath = (string)chromeAppPathKey.GetValue(string.Empty, string.Empty);
          if (!string.IsNullOrEmpty(chromePath) && File.Exists(chromePath))
          {
            return chromePath;
          }
        }
      }

      // Check from registry from LocalMachine
      using (RegistryKey chromeAppPathKey = Registry.LocalMachine.OpenSubKey(ChromeHelper.AppPathRegKey, RegistryKeyPermissionCheck.ReadSubTree))
      {
        if (chromeAppPathKey != null)
        {
          chromePath = (string)chromeAppPathKey.GetValue(string.Empty, string.Empty);
          if (!string.IsNullOrEmpty(chromePath) && File.Exists(chromePath))
          {
            return chromePath;
          }
        }
      }

      // Check hard coded path
      string localApplicationDataPath = Environment.GetFolderPath(Environment.SpecialFolder.LocalApplicationData);
      chromePath = localApplicationDataPath + ChromeHelper.Chrome;
      if (File.Exists(chromePath))
      {
        return chromePath;
      }

      return string.Empty;
    }
  }
}