﻿// ---------------------------------------------------------------
// <copyright file="FormManager.cs" company="B33Rbaron">
//     Copyright (c) Daniel Birler.
//     Licensed under Microsoft Public License (Ms-PL).
// </copyright>
// <author>Daniel Birler</author>
// ---------------------------------------------------------------

namespace ChromeHost
{
  using System;
  using System.Collections.Generic;
  using System.Linq;
  using System.Text;
  using System.Windows.Forms;

  /// <summary>
  /// A class managing the various forms to show.
  /// </summary>
  public sealed class FormManager
  {
    /// <summary>
    /// This field is responsible for initializing the FormManager class.
    /// </summary>
    private static readonly Lazy<FormManager> Lazy = new Lazy<FormManager>(() => new FormManager());

    /// <summary>
    /// The info form.
    /// </summary>
    private InfoForm infoForm = null;

    /// <summary>
    /// The about form.
    /// </summary>
    private AboutForm aboutForm = null;

    /// <summary>
    /// The settings form.
    /// </summary>
    private SettingsForm settingsForm = null;

    /// <summary>
    /// Prevents a default instance of the FormManager class from being created.
    /// </summary>
    private FormManager()
    {
    }

    /// <summary>
    /// Gets the one and only instance of the FormManager class.
    /// </summary>
    public static FormManager Instance
    {
      get { return Lazy.Value; }
    }

    /// <summary>
    /// Shows the settings form.
    /// </summary>
    public void ShowSettingsForm()
    {
      if (this.settingsForm == null)
      {
        this.settingsForm = new SettingsForm();
        this.settingsForm.FormClosed += this.SettingsFormClosed;
      }

      if (this.settingsForm != null && !this.settingsForm.IsDisposed)
      {
        this.settingsForm.Show();
        this.settingsForm.BringToFront();
      }
    }

    /// <summary>
    /// Shows the info form.
    /// </summary>
    public void ShowInfoForm()
    {
      if (this.infoForm == null)
      {
        this.infoForm = new InfoForm();
        this.infoForm.FormClosed += this.InfoFormClosed;
      }

      if (this.infoForm != null && !this.infoForm.IsDisposed)
      {
        this.infoForm.Show();
        this.infoForm.BringToFront();
      }
    }

    /// <summary>
    /// Show the about form.
    /// </summary>
    public void ShowAboutForm()
    {
      if (this.aboutForm == null)
      {
        this.aboutForm = new AboutForm();
        this.aboutForm.FormClosed += this.AboutFormClosed;
      }

      if (this.aboutForm != null && !this.aboutForm.IsDisposed)
      {
        this.aboutForm.Show();
        this.aboutForm.BringToFront();
      }
    }

    /// <summary>
    /// Occurs when the settings form is closed.
    /// </summary>
    /// <param name="sender">The source of the event.</param>
    /// <param name="e">A System.Windows.Forms.FormClosedEventArgs that contains the event data.</param>
    private void SettingsFormClosed(object sender, FormClosedEventArgs e)
    {
      this.settingsForm.FormClosed -= this.SettingsFormClosed;
      this.settingsForm.Dispose();
      this.settingsForm = null;
    }

    /// <summary>
    /// Occurs when the info form is closed.
    /// </summary>
    /// <param name="sender">The source of the event.</param>
    /// <param name="e">A System.Windows.Forms.FormClosedEventArgs that contains the event data.</param>
    private void InfoFormClosed(object sender, FormClosedEventArgs e)
    {
      this.infoForm.FormClosed -= this.InfoFormClosed;
      this.infoForm.Dispose();
      this.infoForm = null;
    }

    /// <summary>
    /// Occurs when the about form is closed.
    /// </summary>
    /// <param name="sender">The source of the event.</param>
    /// <param name="e">A System.Windows.Forms.FormClosedEventArgs that contains the event data.</param>
    private void AboutFormClosed(object sender, FormClosedEventArgs e)
    {
      this.aboutForm.FormClosed -= this.AboutFormClosed;
      this.aboutForm.Dispose();
      this.aboutForm = null;
    }
  }
}