﻿// ---------------------------------------------------------------
// <copyright file="NativeMethods.cs" company="B33Rbaron">
//     Copyright (c) Daniel Birler.
//     Licensed under Microsoft Public License (Ms-PL).
// </copyright>
// <author>Daniel Birler</author>
// ---------------------------------------------------------------

namespace ChromeHost
{
  using System;
  using System.Runtime.InteropServices;
  using System.Text;

  /// <summary>
  /// A class for native operating system constants, structs, classes and methods.
  /// </summary>
  internal static class NativeMethods
  {
    /// <summary>
    /// The WM_CLOSE window message.
    /// </summary>
    internal const uint WM_CLOSE = 0x0010;

    /// <summary>
    /// The WM_QUERYENDSESSION window message.
    /// </summary>
    internal const uint WM_QUERYENDSESSION = 0x0011;

    /// <summary>
    /// The WM_ENDSESSION window message.
    /// </summary>
    internal const uint WM_ENDSESSION = 0x0016;

    /// <summary>
    /// The WM_HOTKEY window message.
    /// </summary>
    internal const uint WM_HOTKEY = 0x0312;

    /// <summary>
    /// The HWND_NOTOPMOST constant.
    /// </summary>
    internal const int HWND_NOTOPMOST = -2;

    /// <summary>
    /// The HWND_TOPMOST constant.
    /// </summary>
    internal const int HWND_TOPMOST = -1;

    /// <summary>
    /// The HWND_TOP constant.
    /// </summary>
    internal const int HWND_TOP = 0;

    /// <summary>
    /// The GWL_EXSTYLE constant.
    /// </summary>
    internal const int GWL_EXSTYLE = -20;

    /// <summary>
    /// The GWL_STYLE constant.
    /// </summary>
    internal const int GWL_STYLE = -16;

    /// <summary>
    /// The WS_EX_TOPMOST constant.
    /// </summary>
    internal const int WS_EX_TOPMOST = 0x00000008;

    /// <summary>
    /// The WS_EX_TOOLWINDOW constant.
    /// </summary>    
    internal const int WS_EX_TOOLWINDOW = 0x00000080;

    /// <summary>
    /// The WS_THICKFRAME constant.
    /// </summary>
    internal const int WS_THICKFRAME = 0x00040000;

    /// <summary>
    /// The WS_DLGFRAME constant.
    /// </summary>
    internal const int WS_DLGFRAME = 0x00400000;

    /// <summary>
    /// The WS_BORDER constant.
    /// </summary>
    internal const int WS_BORDER = 0x00800000;

    /// <summary>
    /// The WS_CHILD constant.
    /// </summary>
    internal const int WS_CHILD = 0x40000000;

    /// <summary>
    /// The SWP_NOSIZE constant.
    /// </summary>
    internal const int SWP_NOSIZE = 0x0001;

    /// <summary>
    /// The SWP_NOMOVE constant.
    /// </summary>
    internal const int SWP_NOMOVE = 0x0002;

    /// <summary>
    /// The SWP_FRAMECHANGED constant.
    /// </summary>
    internal const int SWP_FRAMECHANGED = 0x0020;

    /// <summary>
    /// Indicates that uPosition gives the identifier of the menu item.
    /// </summary>
    internal const uint MF_BYCOMMAND = 0x00000000;

    /// <summary>
    /// The AW_VER_POSITIVE constant.
    /// </summary>
    internal const int AW_VER_POSITIVE = 0x00000004;

    /// <summary>
    /// The AW_VER_NEGATIVE constant.
    /// </summary>
    internal const int AW_VER_NEGATIVE = 0x00000008;

    /// <summary>
    /// The AW_HIDE constant.
    /// </summary>
    internal const int AW_HIDE = 0x00010000;

    /// <summary>
    /// The AW_ACTIVATE constant.
    /// </summary>
    internal const int AW_ACTIVATE = 0x00020000;

    /// <summary>
    /// The callback function is not mapped into the address space of the process that generates
    /// the event. Because the hook function is called across process boundaries, the system must
    /// queue events. Although this method is asynchronous, events are guaranteed to be in sequential
    /// order.
    /// </summary>
    internal const uint WINEVENT_OUTOFCONTEXT = 0x0000;

    /// <summary>
    /// Prevents this instance of the hook from receiving the events that are generated by threads
    /// in this process. This flag does not prevent threads from generating events.
    /// </summary>
    internal const uint WINEVENT_SKIPOWNPROCESS = 0x0002;

    /// <summary>
    /// An object has been created. The system sends this event for the following user interface
    /// elements: caret, header control, list-view control, tab control, toolbar control, tree view
    /// control and window object.
    /// </summary>
    internal const uint EVENT_OBJECT_CREATE = 0x8000;

    /// <summary>
    /// A hidden object is shown. The system sends this event for the following user interface
    /// elements: caret, cursor, and window object.
    /// </summary>
    internal const uint EVENT_OBJECT_SHOW = 0x8002;

    /// <summary>
    /// The window itself rather than a child object.
    /// </summary>
    internal const long OBJID_WINDOW = 0x00000000;

    /// <summary>
    /// Identifies that the event was triggered by itself.
    /// </summary>
    internal const long CHILDID_SELF = 0;

    /// <summary>
    /// An application-defined callback function used with the EnumThreadWindows function. It
    /// receives the window handles associated with a thread. The WNDENUMPROC type defines a
    /// pointer to this callback function. EnumThreadWndProc is a placeholder for the
    /// application-defined function name.
    /// </summary>
    /// <param name="hWnd">A handle to a window associated with the thread specified in the
    /// EnumThreadWindows function.</param>
    /// <param name="lParam">The application-defined value given in the EnumThreadWindows function.</param>
    /// <returns>To continue enumeration, the callback function must return TRUE;
    /// to stop enumeration, it must return FALSE.</returns>
    internal delegate bool EnumThreadWndProc(IntPtr hWnd, IntPtr lParam);

    /// <summary>
    /// An application-defined callback (or hook) function that the system calls in response to
    /// events generated by an accessible object. The hook function processes the event notifications
    /// as required. Clients install the hook function and request specific types of event notifications
    /// by calling SetWinEventHook.
    /// </summary>
    /// <param name="hWinEventHook">Handle to an event hook function. This value is returned by
    /// SetWinEventHook when the hook function is installed and is specific to each instance of
    /// the hook function.</param>
    /// <param name="eventType">Specifies the event that occurred.</param>
    /// <param name="hWnd">Handle to the window that generates the event, or NULL if no window
    /// is associated with the event. For example, the mouse pointer is not associated with a window.</param>
    /// <param name="idObject">Identifies the object associated with the event.</param>
    /// <param name="idChild">Identifies whether the event was triggered by an object or a child element
    /// of the object. If this value is CHILDID_SELF, the event was triggered by the object; otherwise,
    /// this value is the child ID of the element that triggered the event.</param>
    /// <param name="idEventThread">Identifies the thread that generated the event, or the thread
    /// that owns the current window.</param>
    /// <param name="dwmsEventTime">Specifies the time, in milliseconds, that the event was generated.</param>
    internal delegate void WinEventProc(IntPtr hWinEventHook, uint eventType, IntPtr hWnd, long idObject, long idChild, uint idEventThread, uint dwmsEventTime);

    /// <summary>
    /// A enum for keyboard modifiers.
    /// </summary>
    [Flags]
    internal enum MOD : int
    {
      /// <summary>
      /// No modifiers are set.
      /// </summary>
      NONE = 0x0000,

      /// <summary>
      /// The alt modifier is valid.
      /// </summary>
      ALT = 0x0001,

      /// <summary>
      /// The control modifier is valid.
      /// </summary>
      CONTROL = 0x0002,

      /// <summary>
      /// The shift modifier is valid.
      /// </summary>
      SHIFT = 0x0004,

      /// <summary>
      /// The windows modifier is valid.
      /// </summary>
      WIN = 0x0008
    }

    /// <summary>
    /// Retrieves information about the specified window.
    /// </summary>
    /// <param name="hWnd">A handle to the window and, indirectly, the class to which the window belongs.</param>
    /// <param name="nIndex">The zero-based offset to the value to be retrieved.</param>
    /// <returns>If the function succeeds, the return value is the requested value.
    /// If the function fails, the return value is zero.</returns>
    internal static IntPtr GetWindowLongPtr(IntPtr hWnd, int nIndex)
    {
      if (IntPtr.Size == 8)
      {
        return GetWindowLongPtr64(hWnd, nIndex);
      }
      else
      {
        return new IntPtr(GetWindowLong32(hWnd, nIndex));
      }
    }

    /// <summary>
    /// Changes an attribute of the specified window.
    /// </summary>
    /// <param name="hWnd">A handle to the window and, indirectly, the class to which the window belongs.</param>
    /// <param name="nIndex">The zero-based offset to the value to be set.</param>
    /// <param name="dwNewLong">The replacement value.</param>
    /// <returns>If the function succeeds, the return value is the previous value of the specified offset.
    /// If the function fails, the return value is zero.</returns>
    internal static IntPtr SetWindowLongPtr(IntPtr hWnd, int nIndex, IntPtr dwNewLong)
    {
      if (IntPtr.Size == 8)
      {
        return SetWindowLongPtr64(hWnd, nIndex, dwNewLong);
      }
      else
      {
        return new IntPtr(SetWindowLong32(hWnd, nIndex, dwNewLong.ToInt32()));
      }
    }

    /// <summary>
    /// Enumerates all nonchild windows associated with a thread by passing the handle to each window,
    /// in turn, to an application-defined callback function. EnumThreadWindows continues until the
    /// last window is enumerated or the callback function returns FALSE. To enumerate child windows
    /// of a particular window, use the EnumChildWindows function.
    /// </summary>
    /// <param name="dwThreadId">The identifier of the thread whose windows are to be enumerated.</param>
    /// <param name="lpfn">A pointer to an application-defined callback function. For more
    /// information, see EnumThreadWndProc.</param>
    /// <param name="lParam">An application-defined value to be passed to the callback function.</param>
    /// <returns>If the callback function returns TRUE for all windows in the thread specified
    /// by dwThreadId, the return value is TRUE. If the callback function returns FALSE on any
    /// enumerated window, or if there are no windows found in the thread specified by dwThreadId,
    /// the return value is FALSE.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    [return: MarshalAs(UnmanagedType.Bool)]
    internal static extern bool EnumThreadWindows(uint dwThreadId, EnumThreadWndProc lpfn, IntPtr lParam);

    /// <summary>
    /// Registers a system wide hotkey for the window.
    /// </summary>
    /// <param name="hWnd">The handle from the window to register the hotkey.</param>
    /// <param name="id">The id to register for.</param>
    /// <param name="fsModifiers">The key modifiers.</param>
    /// <param name="vk">The key to register.</param>
    /// <returns>True if successful, otherwise false.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    [return: MarshalAs(UnmanagedType.Bool)]
    internal static extern bool RegisterHotKey(IntPtr hWnd, int id, uint fsModifiers, uint vk);

    /// <summary>
    /// Unregisters a system wide hotkey registered to the window.
    /// </summary>
    /// <param name="hWnd">The handle from the window used to register the hotkey.</param>
    /// <param name="id">The id used to register the window.</param>
    /// <returns>True if successful, otherwise false.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    [return: MarshalAs(UnmanagedType.Bool)]
    internal static extern bool UnregisterHotKey(IntPtr hWnd, int id);

    /// <summary>
    /// Retrieves a handle to the foreground window (the window with which the user is currently working).
    /// </summary>
    /// <returns>The return value is a handle to the foreground window.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    internal static extern IntPtr GetForegroundWindow();

    /// <summary>
    /// Retrieves information about the specified window.
    /// </summary>
    /// <param name="hWnd">A handle to the window and, indirectly, the class to which the window belongs.</param>
    /// <param name="nIndex">The zero-based offset to the value to be retrieved.</param>
    /// <returns>If the function succeeds, the return value is the requested value.
    /// If the function fails, the return value is zero.</returns>
    [DllImport("user32.dll", EntryPoint = "GetWindowLongPtr", SetLastError = true, CharSet = CharSet.Unicode)]
    internal static extern IntPtr GetWindowLongPtr64(IntPtr hWnd, int nIndex);

    /// <summary>
    /// Retrieves information about the specified window.
    /// </summary>
    /// <param name="hWnd">A handle to the window and, indirectly, the class to which the window belongs.</param>
    /// <param name="nIndex">The zero-based offset to the value to be retrieved.</param>
    /// <returns>If the function succeeds, the return value is the requested value.
    /// If the function fails, the return value is zero.</returns>
    [DllImport("user32.dll", EntryPoint = "GetWindowLong", SetLastError = true, CharSet = CharSet.Unicode)]
    internal static extern int GetWindowLong32(IntPtr hWnd, int nIndex);

    /// <summary>
    /// Changes an attribute of the specified window.
    /// </summary>
    /// <param name="hWnd">A handle to the window and, indirectly, the class to which the window belongs.</param>
    /// <param name="nIndex">The zero-based offset to the value to be set.</param>
    /// <param name="dwNewLong">The replacement value.</param>
    /// <returns>If the function succeeds, the return value is the previous value of the specified offset.
    /// If the function fails, the return value is zero.</returns>
    [DllImport("user32.dll", EntryPoint = "SetWindowLongPtr", SetLastError = true, CharSet = CharSet.Unicode)]
    internal static extern IntPtr SetWindowLongPtr64(IntPtr hWnd, int nIndex, IntPtr dwNewLong);

    /// <summary>
    /// Changes an attribute of the specified window.
    /// </summary>
    /// <param name="hWnd">A handle to the window and, indirectly, the class to which the window belongs.</param>
    /// <param name="nIndex">The zero-based offset to the value to be set.</param>
    /// <param name="dwNewLong">The replacement value.</param>
    /// <returns>If the function succeeds, the return value is the previous value of the specified offset.
    /// If the function fails, the return value is zero.</returns>
    [DllImport("user32.dll", EntryPoint = "SetWindowLong", SetLastError = true, CharSet = CharSet.Unicode)]
    internal static extern int SetWindowLong32(IntPtr hWnd, int nIndex, int dwNewLong);

    /// <summary>
    /// Changes the size, position, and Z order of a child, pop-up, or top-level window.
    /// </summary>
    /// <param name="hWnd">A handle to the window.</param>
    /// <param name="hWndInsertAfter">A handle to the window to precede the positioned window in the Z order.</param>
    /// <param name="x">The new position of the left side of the window, in client coordinates.</param>
    /// <param name="y">The new position of the top of the window, in client coordinates.</param>
    /// <param name="cx">The new width of the window, in pixels.</param>
    /// <param name="cy">The new height of the window, in pixels.</param>
    /// <param name="uFlags">The window sizing and positioning flags.</param>
    /// <returns>If the function succeeds, the return value is nonzero.
    /// If the function fails, the return value is zero.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    [return: MarshalAs(UnmanagedType.Bool)]
    internal static extern bool SetWindowPos(IntPtr hWnd, IntPtr hWndInsertAfter, int x, int y, int cx, int cy, uint uFlags);

    /// <summary>
    /// Retrieves a handle to the specified window's parent or owner.
    /// </summary>
    /// <param name="hWnd">A handle to the window whose parent window handle is to be retrieved.</param>
    /// <returns>If the window is a child window, the return value is a handle to the parent window. If the window is
    /// a top-level window with the WS_POPUP style, the return value is a handle to the owner window.
    /// If the function fails, the return value is NULL.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    internal static extern IntPtr GetParent(IntPtr hWnd);

    /// <summary>
    /// Changes the parent window of the specified child window.
    /// </summary>
    /// <param name="hWndChild">A handle to the child window.</param>
    /// <param name="hWndNewParent">A handle to the new parent window.</param>
    /// <returns>If the function succeeds, the return value is a handle to the previous parent window.
    /// If the function fails, the return value is NULL.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    internal static extern IntPtr SetParent(IntPtr hWndChild, IntPtr hWndNewParent);

    /// <summary>
    /// Retrieves the name of the class to which the specified window belongs.
    /// </summary>
    /// <param name="hWnd">A handle to the window and, indirectly, the class to which the window belongs.</param>
    /// <param name="lpClassName">The class name string.</param>
    /// <param name="nMaxCount">The length of the lpClassName buffer in characters. The buffer must be large
    /// enough to include the terminating null character; otherwise, the class name string is truncated to
    /// nMaxCount-1 characters.</param>
    /// <returns>If the function succeeds, the return value is the number of characters copied to the buffer,
    /// not including the terminating null character. If the function fails, the return value is zero. To get
    /// extended error information, call GetLastError.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    internal static extern int GetClassName(IntPtr hWnd, StringBuilder lpClassName, int nMaxCount);

    /// <summary>
    /// Determines whether the specified window handle identifies an existing window.
    /// </summary>
    /// <param name="hwnd">A handle to the window to be tested.</param>
    /// <returns>If the window handle identifies an existing window, the return value is nonzero.
    /// If the window handle does not identify an existing window, the return value is zero.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    [return: MarshalAs(UnmanagedType.Bool)]
    internal static extern bool IsWindow(IntPtr hwnd);

    /// <summary>
    /// Retrieves the length, in characters, of the specified window's title bar text (if the window has a title bar).
    /// </summary>
    /// <param name="hWnd">A handle to the window or control.</param>
    /// <returns>If the function succeeds, the return value is the length, in characters, of the text.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    internal static extern int GetWindowTextLength(IntPtr hWnd);

    /// <summary>
    /// Copies the text of the specified window's title bar (if it has one) into a buffer.
    /// </summary>
    /// <param name="hWnd">A handle to the window or control containing the text.</param>
    /// <param name="lpString">The buffer that will receive the text.</param>
    /// <param name="nMaxCount">The maximum number of characters to copy to the buffer, including the null character.</param>
    /// <returns>If the function succeeds, the return value is the length, in characters, of the copied string,
    /// not including the terminating null character. If the window has no title bar or text, if the title bar is empty,
    /// or if the window or control handle is invalid, the return value is zero.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    internal static extern int GetWindowText(IntPtr hWnd, StringBuilder lpString, int nMaxCount);

    /// <summary>
    /// Sets the keyboard focus to the specified window.
    /// </summary>
    /// <param name="hWnd">A handle to the window that will receive the keyboard input.</param>
    /// <returns>If the function succeeds, the return value is the handle to the window that previously had
    /// the keyboard focus. If the hWnd parameter is invalid or the window is not attached to the calling
    /// thread's message queue, the return value is NULL.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    internal static extern IntPtr SetFocus(IntPtr hWnd);

    /// <summary>
    /// Changes the position and dimensions of the specified window. For a top-level window, the position
    /// and dimensions are relative to the upper-left corner of the screen. For a child window, they are
    /// relative to the upper-left corner of the parent window's client area. 
    /// </summary>
    /// <param name="hWnd">A handle to the window.</param>
    /// <param name="X">The new position of the left side of the window.</param>
    /// <param name="Y">The new position of the top of the window.</param>
    /// <param name="nWidth">The new width of the window.</param>
    /// <param name="nHeight">The new height of the window.</param>
    /// <param name="bRepaint">Indicates whether the window is to be repainted.</param>
    /// <returns>If the function succeeds, the return value is nonzero.
    /// If the function fails, the return value is zero.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    [return: MarshalAs(UnmanagedType.Bool)]
    internal static extern bool MoveWindow(IntPtr hWnd, int X, int Y, int nWidth, int nHeight, [MarshalAs(UnmanagedType.Bool)] bool bRepaint);

    /// <summary>
    /// Enables you to produce special effects when showing or hiding windows. There are four types of
    /// animation: roll, slide, collapse or expand, and alpha-blended fade.
    /// </summary>
    /// <param name="hwnd">A handle to the window to animate. The calling thread must own this window.</param>
    /// <param name="dwTime">The time it takes to play the animation, in milliseconds.</param>
    /// <param name="dwFlags">The type of animation.</param>
    /// <returns>f the function succeeds, the return value is nonzero. If the function fails, the return
    /// value is zero.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    [return: MarshalAs(UnmanagedType.Bool)]
    internal static extern bool AnimateWindow(IntPtr hwnd, int dwTime, int dwFlags);

    /// <summary>
    /// Sends the specified message to a window or windows.
    /// </summary>
    /// <param name="hWnd">A handle to the window whose window procedure will receive the message.</param>
    /// <param name="Msg">The message to be sent.</param>
    /// <param name="wParam">Additional message-specific information 1.</param>
    /// <param name="lParam">Additional message-specific information 2.</param>
    /// <returns>The return value specifies the result of the message processing; it depends on the message sent.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    internal static extern IntPtr SendMessage(IntPtr hWnd, uint Msg, IntPtr wParam, IntPtr lParam);

    /// <summary>
    /// Retrieves the identifier of the thread that created the specified window and,
    /// optionally, the identifier of the process that created the window.
    /// </summary>
    /// <param name="hWnd">A handle to the window.</param>
    /// <param name="lpdwProcessId">A pointer to a variable that receives the process identifier.
    /// If this parameter is not NULL, GetWindowThreadProcessId copies the identifier of the process to the variable;
    /// otherwise, it does not.</param>
    /// <returns>The return value is the identifier of the thread that created the window.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    internal static extern uint GetWindowThreadProcessId(IntPtr hWnd, out uint lpdwProcessId);

    /// <summary>
    /// Retrieves the thread identifier of the calling thread.
    /// </summary>
    /// <returns>The return value is the thread identifier of the calling thread.</returns>
    [DllImport("kernel32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    internal static extern uint GetCurrentThreadId();

    /// <summary>
    /// Set the bitmaps of a MenuItem.
    /// </summary>
    /// <param name="hMenu">The handle to the menu.</param>
    /// <param name="uPosition">The position or command.</param>
    /// <param name="uFlags">Specifies if uPosition is a command or position.</param>
    /// <param name="hBitmapUnchecked">The unchecked bitmap.</param>
    /// <param name="hBitmapChecked">The checked bitmap.</param>
    /// <returns>True if successful, otherwise false.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    [return: MarshalAs(UnmanagedType.Bool)]
    internal static extern bool SetMenuItemBitmaps(IntPtr hMenu, uint uPosition, uint uFlags, IntPtr hBitmapUnchecked, IntPtr hBitmapChecked);

    /// <summary>
    /// The DeleteObject function deletes a logical pen, brush, font, bitmap, region, or palette, freeing all system
    /// resources associated with the object. After the object is deleted, the specified handle is no longer valid.
    /// </summary>
    /// <param name="hObject">A handle to a logical pen, brush, font, bitmap, region, or palette.</param>
    /// <returns>If the function succeeds, the return value is nonzero. If the specified handle is not valid
    /// or is currently selected into a DC, the return value is zero.</returns>
    [DllImport("gdi32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    [return: MarshalAs(UnmanagedType.Bool)]
    internal static extern bool DeleteObject(IntPtr hObject);

    /// <summary>
    /// Retrieves the dimensions of the bounding rectangle of the specified window. The dimensions are given
    /// in screen coordinates that are relative to the upper-left corner of the screen.
    /// </summary>
    /// <param name="hWnd">A handle to the window.</param>
    /// <param name="lpRect">A pointer to a RECT structure that receives the screen coordinates of the
    /// upper-left and lower-right corners of the window.</param>
    /// <returns>If the function succeeds, the return value is nonzero.
    /// If the function fails, the return value is zero.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    [return: MarshalAs(UnmanagedType.Bool)]
    internal static extern bool GetWindowRect(IntPtr hWnd, out RECT lpRect);

    /// <summary>
    /// Sets an event hook function for a range of events.
    /// </summary>
    /// <param name="eventMin">Specifies the event constant for the lowest event value in the range of events
    /// that are handled by the hook function. This parameter can be set to EVENT_MIN to indicate the lowest
    /// possible event value.</param>
    /// <param name="eventMax">Specifies the event constant for the highest event value in the range of events
    /// that are handled by the hook function. This parameter can be set to EVENT_MAX to indicate the highest
    /// possible event value.</param>
    /// <param name="hmodWinEventProc">Handle to the DLL that contains the hook function at lpfnWinEventProc,
    /// if the WINEVENT_INCONTEXT flag is specified in the dwFlags parameter. If the hook function is not
    /// located in a DLL, or if the WINEVENT_OUTOFCONTEXT flag is specified, this parameter is NULL.</param>
    /// <param name="pfnWinEventProc">Pointer to the event hook function.</param>
    /// <param name="idProcess">Specifies the ID of the process from which the hook function receives events.
    /// Specify zero (0) to receive events from all processes on the current desktop.</param>
    /// <param name="idThread">Specifies the ID of the thread from which the hook function receives events.
    /// If this parameter is zero, the hook function is associated with all existing threads on the current
    /// desktop.</param>
    /// <param name="dwFlags">Flag values that specify the location of the hook function and of the events
    /// to be skipped.</param>
    /// <returns>If successful, returns an HWINEVENTHOOK value that identifies this event hook instance.
    /// Applications save this return value to use it with the UnhookWinEvent function. If unsuccessful,
    /// returns zero.</returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    internal static extern IntPtr SetWinEventHook(uint eventMin, uint eventMax, IntPtr hmodWinEventProc, WinEventProc pfnWinEventProc, uint idProcess, uint idThread, uint dwFlags);

    /// <summary>
    /// Removes an event hook function created by a previous call to SetWinEventHook.
    /// </summary>
    /// <param name="hWinEventHook">Handle to the event hook returned in the previous call to SetWinEventHook.</param>
    /// <returns>If successful, returns TRUE; otherwise, returns FALSE. </returns>
    [DllImport("user32.dll", SetLastError = true, CharSet = CharSet.Unicode)]
    [return: MarshalAs(UnmanagedType.Bool)]
    internal static extern bool UnhookWinEvent(IntPtr hWinEventHook);

    /// <summary>
    /// The RECT structure defines the coordinates of the upper-left and lower-right corners of a rectangle. 
    /// </summary>
    [StructLayout(LayoutKind.Sequential)]
    internal struct RECT
    {
      /// <summary>
      /// The x position of upper-left corner.
      /// </summary>
      public int Left;

      /// <summary>
      /// The y position of upper-left corner.
      /// </summary>
      public int Top;

      /// <summary>
      /// The x position of lower-right corner.
      /// </summary>
      public int Right;

      /// <summary>
      /// The y position of lower-right corner.
      /// </summary>
      public int Bottom;
    }
  }
}