﻿// ---------------------------------------------------------------
// <copyright file="ChromeHostForm.cs" company="B33Rbaron">
//     Copyright (c) Daniel Birler.
//     Licensed under Microsoft Public License (Ms-PL).
// </copyright>
// <author>Daniel Birler</author>
// ---------------------------------------------------------------

namespace ChromeHost
{
  using System;
  using System.Diagnostics;
  using System.Drawing;
  using System.Globalization;
  using System.Runtime.InteropServices;
  using System.Windows.Forms;

  /// <summary>
  /// This is a class for hosting chrome windows.
  /// </summary>
  public sealed partial class ChromeHostForm : Form
  {
    /// <summary>
    /// The chrome child window handle.
    /// </summary>
    private IntPtr childWindowHandle = IntPtr.Zero;

    /// <summary>
    /// The window styles to set or unset.
    /// </summary>
    private int windowStyleBits = NativeMethods.WS_DLGFRAME | NativeMethods.WS_THICKFRAME | NativeMethods.WS_BORDER | NativeMethods.WS_CHILD;

    /// <summary>
    /// Determines if the window is a topmost window.
    /// </summary>
    private bool isTopMost = false;

    /// <summary>
    /// Determines whether the window has a thick frame.
    /// </summary>
    private bool hasThickFrame = true;

    /// <summary>
    /// Determines whether this ChromeHostForm is an animated console window.
    /// </summary>
    private bool isAnimatedConsole = false;

    /// <summary>
    /// Initializes a new instance of the ChromeHostForm class.
    /// </summary>
    /// <param name="childWindowHandle">The chrome child window handle.</param>
    public ChromeHostForm(IntPtr childWindowHandle)
    {
      this.childWindowHandle = childWindowHandle;
      this.InitializeComponent();
    }

    /// <summary>
    /// Gets a value indicating whether this ChromeHostForm is an animated console window.
    /// </summary>
    public bool IsAnimatedConsole
    {
      get { return this.isAnimatedConsole; }
    }

    /// <summary>
    /// This method makes this form topmost or not topmost.
    /// </summary>
    public void SwitchTopMost()
    {
      this.SwitchFrameInternal(!this.hasThickFrame);
      this.SwitchTopMostInternal(true);
    }

    /// <summary>
    /// This method removes or adds a thick frame to the form.
    /// </summary>
    public void SwitchFrame()
    {
      this.SwitchTopMostInternal(!this.isTopMost);
      this.SwitchFrameInternal(true);
    }

    /// <summary>
    /// Moves the form to the specified screen edge.
    /// </summary>
    public void MoveToScreenEdge()
    {
      Screen activeScreen = Screen.FromPoint(this.Location);
      int left = activeScreen.WorkingArea.X;
      int top = activeScreen.WorkingArea.Y;

      NativeMethods.MoveWindow(this.Handle, left, top, this.Width, this.Height, true);
    }

    /// <summary>
    /// Changes the width of the form to the screen width.
    /// </summary>
    public void SetWidthToScreenWidth()
    {
      Screen activeScreen = Screen.FromPoint(this.Location);
      int width = activeScreen.WorkingArea.Width;

      NativeMethods.MoveWindow(this.Handle, this.Left, this.Top, width, this.Height, true);
    }

    /// <summary>
    /// Animates the form in or out using console style animation.
    /// </summary>
    public void AnimateConsole()
    {
      if (!this.isAnimatedConsole)
      {
        NativeMethods.AnimateWindow(this.Handle, 200, NativeMethods.AW_VER_NEGATIVE | NativeMethods.AW_HIDE);
        this.Hide();
      }
      else
      {
        NativeMethods.AnimateWindow(this.Handle, 200, NativeMethods.AW_VER_POSITIVE | NativeMethods.AW_ACTIVATE);
        this.Show();
      }

      this.isAnimatedConsole = !this.isAnimatedConsole;
      this.Invalidate(true);
    }

    /// <summary>
    /// This method is called when the handle of the form is created.
    /// </summary>
    /// <param name="e">An System.EventArgs that contains the event data.</param>
    protected override void OnHandleCreated(EventArgs e)
    {
      base.OnHandleCreated(e);

      IntPtr windowStyle = NativeMethods.GetWindowLongPtr(this.childWindowHandle, NativeMethods.GWL_STYLE);
      IntPtr newWindowStyle = new IntPtr((long)windowStyle ^ this.windowStyleBits);
      NativeMethods.SetWindowLongPtr(this.childWindowHandle, NativeMethods.GWL_STYLE, newWindowStyle);

      NativeMethods.RECT windowRectangle = new NativeMethods.RECT();
      NativeMethods.GetWindowRect(this.childWindowHandle, out windowRectangle);
      this.Location = new Point(windowRectangle.Left, windowRectangle.Top);
      this.Size = new Size(windowRectangle.Right - windowRectangle.Left, windowRectangle.Bottom - windowRectangle.Top);

      IntPtr result = NativeMethods.SetParent(this.childWindowHandle, this.Handle);
      if (result == IntPtr.Zero)
      {
        Debug.Print("SetParent() returned with IntPtr.Zero!\nWin32 Error Code: " + Marshal.GetLastWin32Error().ToString(CultureInfo.CurrentCulture));
      }

      NativeMethods.MoveWindow(this.childWindowHandle, 0, 0, this.ClientRectangle.Width, this.ClientRectangle.Height, true);
    }

    /// <summary>
    /// This method is called when the form is activated.
    /// </summary>
    /// <param name="e">An System.EventArgs that contains the event data.</param>
    protected override void OnActivated(EventArgs e)
    {
      base.OnActivated(e);
      NativeMethods.SetFocus(this.childWindowHandle);
    }

    /// <summary>
    /// This method is called when the form is closed.
    /// </summary>
    /// <param name="e">An System.Windows.Forms.FormClosingEventArgs that contains the event data.</param>
    protected override void OnFormClosing(FormClosingEventArgs e)
    {
      base.OnFormClosing(e);
      NativeMethods.SendMessage(this.childWindowHandle, NativeMethods.WM_CLOSE, IntPtr.Zero, IntPtr.Zero);
    }

    /// <summary>
    /// This method is called when the size of the form has changed.
    /// </summary>
    /// <param name="e">An System.EventArgs that contains the event data.</param>
    protected override void OnSizeChanged(EventArgs e)
    {
      base.OnSizeChanged(e);

      if (this.ClientRectangle.Width > 0 || this.ClientRectangle.Height > 0)
      {
        NativeMethods.MoveWindow(this.childWindowHandle, 0, 0, this.ClientRectangle.Width, this.ClientRectangle.Height, true);
      }
    }

    /// <summary>
    /// Implements switching topmost of the window.
    /// </summary>
    /// <param name="canSwitch">Determines whether top most can be switched.</param>
    private void SwitchTopMostInternal(bool canSwitch)
    {
      if (canSwitch)
      {
        this.isTopMost = !this.isTopMost;

        IntPtr insertAfterWindow = this.isTopMost ? new IntPtr(NativeMethods.HWND_TOPMOST) : new IntPtr(NativeMethods.HWND_NOTOPMOST);
        IntPtr windowStyle = NativeMethods.GetWindowLongPtr(this.Handle, NativeMethods.GWL_STYLE);
        IntPtr newWindowStyle = new IntPtr((long)windowStyle ^ NativeMethods.WS_DLGFRAME);

        NativeMethods.SetWindowLongPtr(this.Handle, NativeMethods.GWL_STYLE, newWindowStyle);
        NativeMethods.SetWindowPos(this.Handle, insertAfterWindow, 0, 0, 0, 0, NativeMethods.SWP_FRAMECHANGED | NativeMethods.SWP_NOMOVE | NativeMethods.SWP_NOSIZE);
      }
    }

    /// <summary>
    /// Implements switching the frame of the window.
    /// </summary>
    /// <param name="canSwitch">Determines whether the frame can be switched.</param>
    private void SwitchFrameInternal(bool canSwitch)
    {
      if (this.isTopMost && canSwitch)
      {
        this.hasThickFrame = !this.hasThickFrame;
        IntPtr windowStyle = NativeMethods.GetWindowLongPtr(this.Handle, NativeMethods.GWL_STYLE);
        IntPtr newWindowStyle = new IntPtr((long)windowStyle ^ NativeMethods.WS_THICKFRAME);
        NativeMethods.SetWindowLongPtr(this.Handle, NativeMethods.GWL_STYLE, newWindowStyle);
      }
    }
  }
}