﻿// ---------------------------------------------------------------
// <copyright file="DetectorWindow.cs" company="B33Rbaron">
//     Copyright (c) Daniel Birler.
//     Licensed under Microsoft Public License (Ms-PL).
// </copyright>
// <author>Daniel Birler</author>
// ---------------------------------------------------------------

namespace ChromeHost
{
  using System;
  using System.Diagnostics;
  using System.Globalization;
  using System.Runtime.InteropServices;
  using System.Windows.Forms;

  /// <summary>
  /// A class derived from NativeWindow to capture Windows System messages.
  /// </summary>
  public sealed class DetectorWindow : NativeWindow, IDisposable
  {
    /// <summary>
    /// The id registered with the first hotkey.
    /// </summary>
    private const int HotKeyIdFirst = 0x0001;

    /// <summary>
    /// The id registered with the second hotkey.
    /// </summary>
    private const int HotKeyIdSecond = 0x0002;

    /// <summary>
    /// The id registered with the third hotkey.
    /// </summary>
    private const int HotKeyIdThird = 0x0003;

    /// <summary>
    /// The id registered with the fourth hotkey.
    /// </summary>
    private const int HotKeyIdFourth = 0x0004;

    /// <summary>
    /// The id registered with the console hotkey.
    /// </summary>
    private const int HotKeyIdConsole = 0x0005;

    /// <summary>
    /// The id registered with the console hotkey for us keyboards.
    /// </summary>
    private const int HotKeyIdConsoleUS = 0x0006;

    /// <summary>
    /// Determines whether the object has been disposed.
    /// </summary>
    private bool isDisposed = false;

    /// <summary>
    /// The first HotKeyModifierPair.
    /// </summary>
    private HotKeyModifierPair hotKeyModifierPairFirst = null;

    /// <summary>
    /// The second HotKeyModifierPair.
    /// </summary>
    private HotKeyModifierPair hotKeyModifierPairSecond = null;

    /// <summary>
    /// The third HotKeyModifierPair.
    /// </summary>
    private HotKeyModifierPair hotKeyModifierPairThird = null;

    /// <summary>
    /// The fourth HotKeyModifierPair.
    /// </summary>
    private HotKeyModifierPair hotKeyModifierPairFourth = null;

    /// <summary>
    /// The console HotKeyModifierPair.
    /// </summary>
    private HotKeyModifierPair hotKeyModifierPairConsole = null;

    /// <summary>
    /// The console HotKeyModifierPair.
    /// </summary>
    private HotKeyModifierPair hotKeyModifierPairConsoleUS = null;

    /// <summary>
    /// The ChromeHostForms that were opened.
    /// </summary>
    private ChromeHostFormList chromeHostFormList = new ChromeHostFormList();

    /// <summary>
    /// Initializes a new instance of the DetectorWindow class.
    /// </summary>
    public DetectorWindow()
    {
      this.CreateHandle(new CreateParams());

      this.hotKeyModifierPairFirst = new HotKeyModifierPair(DetectorWindow.HotKeyIdFirst, Keys.F1, NativeMethods.MOD.CONTROL | NativeMethods.MOD.WIN);
      this.InstallHotKey(this.hotKeyModifierPairFirst);

      this.hotKeyModifierPairSecond = new HotKeyModifierPair(DetectorWindow.HotKeyIdSecond, Keys.F2, NativeMethods.MOD.CONTROL | NativeMethods.MOD.WIN);
      this.InstallHotKey(this.hotKeyModifierPairSecond);

      this.hotKeyModifierPairThird = new HotKeyModifierPair(DetectorWindow.HotKeyIdThird, Keys.F3, NativeMethods.MOD.CONTROL | NativeMethods.MOD.WIN);
      this.InstallHotKey(this.hotKeyModifierPairThird);

      this.hotKeyModifierPairFourth = new HotKeyModifierPair(DetectorWindow.HotKeyIdFourth, Keys.F4, NativeMethods.MOD.CONTROL | NativeMethods.MOD.WIN);
      this.InstallHotKey(this.hotKeyModifierPairFourth);

      this.hotKeyModifierPairConsole = new HotKeyModifierPair(DetectorWindow.HotKeyIdConsole, Keys.OemPipe, NativeMethods.MOD.CONTROL);
      this.InstallHotKey(this.hotKeyModifierPairConsole);

      this.hotKeyModifierPairConsoleUS = new HotKeyModifierPair(DetectorWindow.HotKeyIdConsoleUS, Keys.Oemtilde, NativeMethods.MOD.CONTROL);
      this.InstallHotKey(this.hotKeyModifierPairConsoleUS);

      ChromeWindowWatcher.Instance.WindowCreated += this.OnChromeWindowWatcherWindowCreated;
    }

    /// <summary>
    /// Implementing from IDisposable.Dispose.
    /// </summary>
    public void Dispose()
    {
      this.Dispose(true);
    }

    /// <summary>
    /// This function receives all the windows messages.
    /// </summary>
    /// <param name="m">The message which is currently recieved.</param>
    protected override void WndProc(ref Message m)
    {
      if (m.Msg == NativeMethods.WM_HOTKEY)
      {
        if (m.WParam.ToInt32() == this.hotKeyModifierPairFirst.HotKeyId)
        {
          this.OnHotKeyFirst(IntPtr.Zero);
        }
        else if (m.WParam.ToInt32() == this.hotKeyModifierPairSecond.HotKeyId)
        {
          this.OnHotKeySecond();
        }
        else if (m.WParam.ToInt32() == this.hotKeyModifierPairThird.HotKeyId)
        {
          this.OnHotKeyThird();
        }
        else if (m.WParam.ToInt32() == this.hotKeyModifierPairFourth.HotKeyId)
        {
          this.OnHotKeyFourth();
        }
        else if (m.WParam.ToInt32() == this.hotKeyModifierPairConsole.HotKeyId)
        {
          this.OnHotKeyConsole();
        }
        else if (m.WParam.ToInt32() == this.hotKeyModifierPairConsoleUS.HotKeyId)
        {
          this.OnHotKeyConsole();
        }
      }

      base.WndProc(ref m);
    }

    /// <summary>
    /// Before the DetectorWindow can receive any hotkey messages, it must register the hotkey.
    /// </summary>
    /// <param name="hotKeyModifierPair">The HotKeyModifierPair containing information what to register.</param>
    private void InstallHotKey(HotKeyModifierPair hotKeyModifierPair)
    {
      bool succeed = NativeMethods.RegisterHotKey(this.Handle, hotKeyModifierPair.HotKeyId, (uint)hotKeyModifierPair.HotKeyModifier, (uint)hotKeyModifierPair.HotKey);
      if (!succeed)
      {
        Debug.Print("RegisterHotkey() returned with false!\nWin32 Error Code: " + Marshal.GetLastWin32Error().ToString(CultureInfo.CurrentCulture));
      }
    }

    /// <summary>
    /// It is essential to unregister the hotkey before the application exits.
    /// </summary>
    /// <param name="hotKeyModifierPair">The HotKeyModifierPair containing information what to unregister</param>
    private void UninstallHotKey(HotKeyModifierPair hotKeyModifierPair)
    {
      bool succeed = NativeMethods.UnregisterHotKey(this.Handle, hotKeyModifierPair.HotKeyId);
      if (!succeed)
      {
        Debug.Print("RegisterHotkey() returned with false!\nWin32 Error Code: " + Marshal.GetLastWin32Error().ToString(CultureInfo.CurrentCulture));
      }
    }

    /// <summary>
    /// Retrieves the foreground ChromeHostForm.
    /// </summary>
    /// <returns>Returns the ChromeHostForm if the foreground window is one, otherwise null.</returns>
    private ChromeHostForm RetrieveForegroundChromeHostForm()
    {
      ChromeHostForm chromeHostForm = null;
      IntPtr foregroundWindow = NativeMethods.GetForegroundWindow();
      if (foregroundWindow != IntPtr.Zero)
      {
        chromeHostForm = this.chromeHostFormList.FindFromWindowHandle(foregroundWindow);
      }

      return chromeHostForm;
    }

    /// <summary>
    /// This method is called when a registered hotkey is pressed.
    /// </summary>
    /// <param name="windowHandle">The window handle to operate on, if IntPtr.Zero is passed,
    /// the window handle of the foreground window is used.</param>
    private void OnHotKeyFirst(IntPtr windowHandle)
    {
      IntPtr foregroundWindow = windowHandle != IntPtr.Zero ? windowHandle : NativeMethods.GetForegroundWindow();
      if (foregroundWindow != IntPtr.Zero)
      {
        ChromeHostForm chromeHostForm = this.chromeHostFormList.FindFromWindowHandle(foregroundWindow);
        if (chromeHostForm == null)
        {
          uint processId = 0;
          uint threadId = NativeMethods.GetWindowThreadProcessId(foregroundWindow, out processId);
          if (threadId > 0)
          {
            Process process = Process.GetProcessById((int)processId);
            if (process != null && process.ProcessName.ToUpperInvariant().Equals("CHROME"))
            {
              chromeHostForm = this.chromeHostFormList.AddFromWindowHandle(foregroundWindow);
              chromeHostForm.Activated += this.ChromeHostFormActivated;
              chromeHostForm.FormClosed += this.ChromeHostFormFormClosed;
              chromeHostForm.Show();
            }
          }
        }
        else
        {
          chromeHostForm.SwitchTopMost();
        }
      }
    }

    /// <summary>
    /// This method is called when a registered hotkey is pressed.
    /// </summary>
    private void OnHotKeySecond()
    {
      ChromeHostForm chromeHostForm = this.RetrieveForegroundChromeHostForm();
      if (chromeHostForm != null)
      {
        chromeHostForm.SwitchFrame();
      }
    }

    /// <summary>
    /// This method is called when a registered hotkey is pressed.
    /// </summary>
    private void OnHotKeyThird()
    {
      ChromeHostForm chromeHostForm = this.RetrieveForegroundChromeHostForm();
      if (chromeHostForm != null)
      {
        chromeHostForm.MoveToScreenEdge();
      }
    }

    /// <summary>
    /// This method is called when a registered hotkey is pressed.
    /// </summary>
    private void OnHotKeyFourth()
    {
      ChromeHostForm chromeHostForm = this.RetrieveForegroundChromeHostForm();
      if (chromeHostForm != null)
      {
        chromeHostForm.SetWidthToScreenWidth();
      }
    }

    /// <summary>
    /// This method is called when a registered hotkey is pressed.
    /// </summary>
    private void OnHotKeyConsole()
    {
      foreach (ChromeHostForm chromeHostForm in this.chromeHostFormList)
      {
        if (chromeHostForm.IsAnimatedConsole)
        {
          chromeHostForm.AnimateConsole();
          return;
        }
      }

      if (this.chromeHostFormList.Count >= 1)
      {
        this.chromeHostFormList[0].AnimateConsole();
      }
    }

    /// <summary>
    /// Called when the ChromeWindowWatcher detected that Chrome a window is created.
    /// </summary>
    /// <param name="sender">The source of the event.</param>
    /// <param name="e">A ChromeWindowWatcher.WindowCreatedEventArgs that contains event data.</param>
    private void OnChromeWindowWatcherWindowCreated(object sender, ChromeWindowWatcher.WindowCreatedEventArgs e)
    {
      this.OnHotKeyFirst(e.WindowHandle);
      ChromeWindowWatcher.Instance.Enabled = false;
    }

    /// <summary>
    /// Occurs when a ChromeHostForm is activated.
    /// </summary>
    /// <param name="sender">The source of the event.</param>
    /// <param name="e">An System.EventArgs that contains the event data.</param>
    private void ChromeHostFormActivated(object sender, EventArgs e)
    {
      ChromeHostForm chromeHostForm = sender as ChromeHostForm;
      if (chromeHostForm != null)
      {
        this.chromeHostFormList.MoveToFirstPosition(chromeHostForm);
      }
    }

    /// <summary>
    /// Occurs when a ChromeHostForm is closed.
    /// </summary>
    /// <param name="sender">The source of the event.</param>
    /// <param name="e">A System.Windows.Forms.FormClosedEventArgs that contains the event data.</param>
    private void ChromeHostFormFormClosed(object sender, FormClosedEventArgs e)
    {
      ChromeHostForm chromeHostForm = sender as ChromeHostForm;
      if (chromeHostForm != null)
      {
        chromeHostForm.Activated -= this.ChromeHostFormActivated;
        chromeHostForm.FormClosed -= this.ChromeHostFormFormClosed;
        this.chromeHostFormList.Remove(chromeHostForm);
      }
    }

    /// <summary>
    /// Implementation of the dispose pattern.
    /// </summary>
    /// <param name="isDisposing">Determines whether the object is disposing.</param>
    private void Dispose(bool isDisposing)
    {
      if (!this.isDisposed)
      {
        if (isDisposing)
        {
          for (int i = 0; i < this.chromeHostFormList.Count; i++)
          {
            ChromeHostForm chromeHostForm = this.chromeHostFormList[i];
            chromeHostForm.Close();
          }

          this.chromeHostFormList.Clear();
          this.UninstallHotKey(this.hotKeyModifierPairFirst);
          this.UninstallHotKey(this.hotKeyModifierPairSecond);
          this.UninstallHotKey(this.hotKeyModifierPairThird);
          this.UninstallHotKey(this.hotKeyModifierPairFourth);
          this.UninstallHotKey(this.hotKeyModifierPairConsole);
          this.UninstallHotKey(this.hotKeyModifierPairConsoleUS);

          ChromeWindowWatcher.Instance.WindowCreated -= this.OnChromeWindowWatcherWindowCreated;
        }

        this.isDisposed = true;
      }
    }
  }
}