﻿// ---------------------------------------------------------------
// <copyright file="SettingsForm.cs" company="B33Rbaron">
//     Copyright (c) Daniel Birler.
//     Licensed under Microsoft Public License (Ms-PL).
// </copyright>
// <author>Daniel Birler</author>
// ---------------------------------------------------------------

namespace ChromeHost
{
  using System;
  using System.Collections.ObjectModel;
  using System.Windows.Forms;

  /// <summary>
  /// A form for displaying settings.
  /// </summary>
  public sealed partial class SettingsForm : Form
  {
    /// <summary>
    /// Initializes a new instance of the SettingsForm class.
    /// </summary>
    public SettingsForm()
    {
      this.InitializeComponent();

      Collection<NameUrlPair> nameUrlPairCollection = NameUrlPairManager.Instance.LoadFromSettings();
      foreach (NameUrlPair nameUrlPair in nameUrlPairCollection)
      {
        ListViewItem listViewItem = new ListViewItem(nameUrlPair.Name);
        listViewItem.SubItems.Add(nameUrlPair.Url.OriginalString);

        this.listView1.Items.Add(listViewItem);
      }
    }

    /// <summary>
    /// Raises the System.Windows.Forms.Form.FormClosing event.
    /// </summary>
    /// <param name="e">A System.Windows.Forms.FormClosingEventArgs that contains the event data.</param>
    protected override void OnFormClosing(FormClosingEventArgs e)
    {
      Collection<NameUrlPair> nameUrlPairCollection = new Collection<NameUrlPair>();

      foreach (ListViewItem listviewItem in this.listView1.Items)
      {
        NameUrlPair nameUrlPair = new NameUrlPair(listviewItem.Text, new Uri(listviewItem.SubItems[1].Text, UriKind.RelativeOrAbsolute));
        nameUrlPairCollection.Add(nameUrlPair);
      }

      NameUrlPairManager.Instance.SaveToSettings(nameUrlPairCollection);

      base.OnFormClosing(e);
    }

    /// <summary>
    /// Occurs when the add button is clicked and adds a new ListViewItem.
    /// </summary>
    /// <param name="sender">The source of the event.</param>
    /// <param name="e">A System.EventArgs that contains the event data.</param>
    private void AddClicked(object sender, EventArgs e)
    {
      if (this.listView1.Items.Count < 15)
      {
        ListViewItem listViewItem = new ListViewItem("Google");
        listViewItem.SubItems.Add("http://www.google.de");

        this.listView1.Items.Add(listViewItem);
      }
      else
      {
        using (new DialogCenteringService(this))
        {
          MessageBox.Show(this, "Only 15 entries allowed.", "ChromeHost", MessageBoxButtons.OK, MessageBoxIcon.Warning, MessageBoxDefaultButton.Button1, 0);
        }
      }
    }

    /// <summary>
    /// Occurs when the remove button is clicked.
    /// </summary>
    /// <param name="sender">The source of the event.</param>
    /// <param name="e">A System.EventArgs that contains the event data.</param>
    private void RemoveClicked(object sender, EventArgs e)
    {
      if (this.listView1.SelectedItems.Count == 1)
      {
        this.listView1.Items.Remove(this.listView1.SelectedItems[0]);
      }
    }

    /// <summary>
    /// Occurs when the save button is clicked.
    /// </summary>
    /// <param name="sender">The source of the event.</param>
    /// <param name="e">A System.EventArgs that contains the event data.</param>
    private void SaveClicked(object sender, EventArgs e)
    {
      if (this.listView1.SelectedItems.Count == 1)
      {
        this.listView1.SelectedItems[0].Text = this.tbName.Text;
        this.listView1.SelectedItems[0].SubItems[1].Text = this.tbURL.Text;
        this.listView1.SelectedItems.Clear();
      }
    }

    /// <summary>
    /// Occurs when the Ok Button is clicked.
    /// </summary>
    /// <param name="sender">The source of the event.</param>
    /// <param name="e">A System.EventArgs that contains the event data.</param>
    private void OkClicked(object sender, EventArgs e)
    {
      this.Close();
    }

    /// <summary>
    /// Occurs when the selected index of the ListView changed.
    /// </summary>
    /// <param name="sender">The source of the event.</param>
    /// <param name="e">A System.EventArgs that contains the event data.</param>
    private void SelectedIndexChanged(object sender, EventArgs e)
    {
      if (this.listView1.SelectedItems.Count == 1)
      {
        this.tbName.Text = this.listView1.SelectedItems[0].Text;
        this.tbURL.Text = this.listView1.SelectedItems[0].SubItems[1].Text;
      }
      else
      {
        this.tbName.Text = string.Empty;
        this.tbURL.Text = string.Empty;
      }
    }
  }
}